<?php declare(strict_types=1);

namespace Qu\Postaffiliatepro\Subscriber;

use Shopware\Core\Checkout\Cart\Exception\OrderNotFoundException;
use Shopware\Core\Checkout\Order\Event\OrderStateMachineStateChangeEvent;
use Shopware\Core\Checkout\Order\OrderEntity;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Event\EntityLoadedEvent;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\System\StateMachine\Event\StateMachineStateChangeEvent;
use Shopware\Recovery\Common\HttpClient\CurlClient;
use Shopware\Storefront\Pagelet\Footer\FooterPageletLoadedEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Qu\Postaffiliatepro\Service\HttpService;
use Qu\Postaffiliatepro\Service\OrderService;
use Qu\Postaffiliatepro\Service\SettingsService;

class EventSubscriber implements EventSubscriberInterface {

    /**
     * @var OrderService
     */
    private $orderService;

    /**
     * @var HttpService
     */
    private $httpService;

    /**
     * @var SettingsService
     */
    private $config;

    public function __construct(
        SettingsService $config,
        OrderService $orderService,
        HttpService $httpService
    ) {
        $this->config = $config;
        $this->orderService = $orderService;
        $this->httpService = $httpService;
    }

    public static function getSubscribedEvents(): array {
        // Return the events to listen to as an array like this:  <event to listen to> => <method to execute>
        return [
          FooterPageletLoadedEvent::class => 'onFooterDisplay',
          'state_machine.order.state_changed' => 'onOrderStateChange'
        ];
    }

    /**
     * @throws OrderNotFoundException
     */
    public function onOrderStateChange(StateMachineStateChangeEvent $event): bool {
        $transitionId = $event->getTransition()->getEntityId();
        $order = $this->orderService->getOrder($transitionId, $event->getContext());
        $orderId = $order->getOrderNumber();

        $this->orderService->debug('Order ID '.$orderId.' just changed status to '.$event->getStateName());

        // translate status
        $status = 'P';
        switch ($event->getStateName()) {
            case 'cancelled': $status = 'D'; break;
            case 'completed': $status = 'A'; break;
        }

        if ($status == 'P') {
            $this->orderService->debug('No need for changes now.');
            return false;
        }

        // call PAP
        $config = $this->config->getSettings();
        $url = $config->getPapUrl();
        $username = $config->getUsername();
        $pass = $config->getPassword();

        if (empty($url) || empty($username) || empty($pass)) {
            $this->orderService->debug('The plugin is not configured.');
            return false;
        }
        
        $body = 'D={"C":"Gpf_Api_AuthService","M":"authenticate","fields":[["name","value","values","error"],["username","'.$username.'",null,""],["password","'.$pass.'",null,""],["roleType","M",null,""],["isFromApi","Y",null,""],["apiVersion","",null,""]]}';
        $response = $this->connectExternal($this->sanitizeURL($url).'scripts/server.php', $body, true);

        if (!empty($response) && $response->success !== 'Y') {
            $this->orderService->debug('We could not connect with API, ending.');
            return false;
        }

        // get session value
        $session = '';
        foreach ($response->fields as $field) {
            if ($field[0] == 'S') {
                $session = $field[1];
            }
        }
        if (empty($session)) {
            $this->orderService->debug('No session for API communication found, ending.');
            return false;
        }

        // load commissions for the order ID
        $body = 'D={"C":"Gpf_Rpc_Server", "M":"run", "requests":[{"C":"Pap_Merchants_Transaction_TransactionsGrid", "M":"getRows", "sort_col":"dateinserted", "sort_asc":false, "limit":50, "filters":[["t_orderid","L","'.$orderId.'(%"]], "columns":[["id"],["id"],["commission"],["totalcost"],["t_orderid"],["productid"],["dateinserted"],["name"],["rtype"],["tier"],["commissionTypeName"],["rstatus"],["payoutstatus"],["firstname"],["lastname"],["userid"],["channel"],["actions"]]}], "S":"'.$session.'"}';
        $response = $this->connectExternal($this->sanitizeURL($url).'scripts/server.php', $body, true);

        if (empty($response)) {
            $this->orderService->debug('API TransactionsGrid problem, ending.');
            return false;
        }

        $ids = '';
        $rstatusKey = '';
        foreach ($response[0]->rows as $row) {
            if ($row[0] == 'id') {
                foreach ($row as $key => $name) {
                    if ($name == 'rstatus') {
                        $rstatusKey = $key;
                        break;
                    }
                }
                continue;
            }
            if ($row[$rstatusKey] == 'P') {
                $ids .= '"'.$row[0].'",';
            }
            if ($status == 'D' && $row[$rstatusKey] == 'A') {
                $ids .= '"'.$row[0].'",';
            }
        }
        if (substr($ids,-1) == ',') {
            $ids = substr($ids,0,-1);
        }

        $body = 'D={"C":"Gpf_Rpc_Server", "M":"run", "requests":[{"C":"Pap_Merchants_Transaction_TransactionsForm", "M":"changeStatus", "merchant_note":"status changed automatically", "status":"'.$status.'", "ids":['.$ids.']}], "S":"'.$session.'"}';
        $response = $this->connectExternal($this->sanitizeURL($url).'scripts/server.php', $body, true);
        $this->orderService->debug('Finished.');
        return true;
    }

    public function onFooterDisplay(FooterPageletLoadedEvent $event): void {
        $config = $this->config->getSettings();
        $event->getPagelet()->addExtension('papPluginConfig', $config);
    }

    private function connectExternal($url, $body) {
        try {
            $this->orderService->debug('POST to URL: '.$url);
            $response = $this->httpService->call($url, $body);
        } catch (Exception $e) {
            $this->orderService->debug('API connection problem: '.$e->getMessage());
        }

        $this->orderService->debug("API response: \n".$response);
        $result = null;
        try {
            $result = json_decode($response);
        } catch (Exception $e) {
        }
        return $result;
    }

    private function sanitizeURL($url): string {
        if ((strpos($url, 'http://') !== false) && (strpos($url, 'https://') !== false)) {
            $url = 'http://'.$url;
        }

        if (substr($url, -1) != '/') {
            $url .= '/';
        }
        return $url;
    }
}